'use client';

import React, { useState, useEffect } from 'react';
import { useParams } from 'next/navigation';
import { 
  User, 
  Plane, 
  Calendar, 
  Clock, 
  MapPin, 
  CreditCard, 
  FileText, 
  CheckCircle,
  XCircle,
  Edit,
  Download
} from 'lucide-react';
import DetailLayout from '@/components/ui/DetailLayout';
import SuggestionsPanel from '@/components/ui/SuggestionsPanel';
import { renderCustomer, renderStatusBadge, ticketStatusConfig } from '@/components/ui/TableRenderers';
import type { TicketRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

export default function TicketDetailPage() {
  return (
    <PermissionGuard requiredPermissions={['tickets_view', 'tickets_view_own', 'tickets_view_all']}>
      <TicketDetailPageContent />
    </PermissionGuard>
  );
}

function TicketDetailPageContent() {
  const params = useParams();
  const ticketId = params.id as string;
  
  const [ticket, setTicket] = useState<TicketRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [comments, setComments] = useState<Comment[]>([]);
  const [flightSuggestions, setFlightSuggestions] = useState<any[]>([]);
  const [suggestionsLoading, setSuggestionsLoading] = useState(false);

  useEffect(() => {
    fetchTicketDetails();
    fetchComments();
    fetchFlightSuggestions();
  }, [ticketId]);

  const fetchTicketDetails = async () => {
    try {
      setLoading(true);
      
      // Mock data - gerçek API ile değiştirilecek
      const mockTicket: TicketRequest = {
        id: ticketId,
        customerId: 'c1',
        customer: {
          id: 'c1',
          type: 'individual',
          firstName: 'Ahmet',
          lastName: 'Yılmaz',
          phone: '+905551234567',
          email: 'ahmet@email.com',
          isVip: false,
          isActive: true,
          createdAt: new Date('2025-01-10'),
          updatedAt: new Date('2025-01-12')
        },
        passengers: [
          {
            id: 'p1',
            firstName: 'Ahmet',
            lastName: 'Yılmaz',
            phone: '+905551234567',
            email: 'ahmet@email.com'
          }
        ],
        airline: 'THY',
        flightNumber: 'TK1234',
        departureAirport: 'IST',
        arrivalAirport: 'ADB',
        departureDate: new Date('2025-01-15'),
        departureTime: '14:30',
        status: 'pending',
        requestDate: new Date('2025-01-10'),
        ticketPrice: 850,
        currency: 'TRY',
        agentNotes: 'Müşteri acil bilet talep ediyor',
        createdAt: new Date('2025-01-10'),
        updatedAt: new Date('2025-01-12')
      };
      
      setTicket(mockTicket);
    } catch (error) {
      console.error('Error fetching ticket details:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchComments = async () => {
    // Mock comments
    const mockComments: Comment[] = [
      {
        id: '1',
        author: 'Sistem',
        content: 'Bilet talebi oluşturuldu',
        timestamp: new Date('2025-01-10T10:00:00'),
        type: 'system'
      },
      {
        id: '2',
        author: 'Ahmet Yılmaz',
        content: 'Merhaba, biletimi ne zaman alabilirim?',
        timestamp: new Date('2025-01-10T14:30:00'),
        type: 'comment'
      },
      {
        id: '3',
        author: 'Agent Mehmet',
        content: 'Merhaba Ahmet Bey, biletinizi bugün içerisinde hazırlayacağız.',
        timestamp: new Date('2025-01-10T15:00:00'),
        type: 'comment'
      },
      {
        id: '4',
        author: 'Agent Mehmet',
        content: 'Müşteri acil seyahat planlıyor, öncelik verilmeli',
        timestamp: new Date('2025-01-10T15:05:00'),
        type: 'comment',
        isInternal: true
      },
      {
        id: '5',
        author: 'Sistem',
        content: 'Durum "Bekliyor" olarak güncellendi',
        timestamp: new Date('2025-01-12T09:00:00'),
        type: 'status_change'
      }
    ];
    
    setComments(mockComments);
  };

  const fetchFlightSuggestions = async () => {
    try {
      setSuggestionsLoading(true);
      
      // Mock flight suggestions
      const mockSuggestions = [
        {
          id: '1',
          airline: 'THY',
          flightNumber: 'TK1234',
          departure: 'IST',
          arrival: 'ADB',
          departureTime: '14:30',
          arrivalTime: '16:00',
          price: 850,
          currency: 'TRY',
          duration: '1s 30dk',
          stops: 0
        },
        {
          id: '2',
          airline: 'PC',
          flightNumber: 'PC2156',
          departure: 'SAW',
          arrival: 'ADB',
          departureTime: '16:45',
          arrivalTime: '18:20',
          price: 720,
          currency: 'TRY',
          duration: '1s 35dk',
          stops: 0
        },
        {
          id: '3',
          airline: 'AJET',
          flightNumber: 'VF3421',
          departure: 'IST',
          arrival: 'ADB',
          departureTime: '19:15',
          arrivalTime: '20:50',
          price: 680,
          currency: 'TRY',
          duration: '1s 35dk',
          stops: 0
        }
      ];
      
      setFlightSuggestions(mockSuggestions);
    } catch (error) {
      console.error('Error fetching flight suggestions:', error);
    } finally {
      setSuggestionsLoading(false);
    }
  };

  const handleAddComment = (content: string, isInternal: boolean) => {
    const newComment: Comment = {
      id: Date.now().toString(),
      author: 'Agent Mehmet', // Gerçek uygulamada current user
      content,
      timestamp: new Date(),
      type: 'comment',
      isInternal
    };
    
    setComments(prev => [...prev, newComment]);
  };

  const handleStatusChange = (newStatus: string) => {
    if (!ticket) return;
    
    setTicket(prev => prev ? { ...prev, status: newStatus as any, updatedAt: new Date() } : null);
    
    // Add system comment for status change
    const statusComment: Comment = {
      id: Date.now().toString(),
      author: 'Sistem',
      content: `Durum "${getStatusLabel(newStatus)}" olarak güncellendi`,
      timestamp: new Date(),
      type: 'status_change'
    };
    
    setComments(prev => [...prev, statusComment]);
  };

  const getStatusLabel = (status: string) => {
    const config = ticketStatusConfig[status];
    return config ? config.label : status;
  };

  const handleSelectFlight = (flight: any) => {
    console.log('Selected flight:', flight);
    
    // Update ticket with selected flight
    if (ticket) {
      setTicket(prev => prev ? {
        ...prev,
        airline: flight.airline,
        flightNumber: flight.flightNumber,
        departureTime: flight.departureTime,
        ticketPrice: flight.price,
        updatedAt: new Date()
      } : null);
      
      // Add comment about flight selection
      const flightComment: Comment = {
        id: Date.now().toString(),
        author: 'Agent Mehmet',
        content: `Uçuş seçildi: ${flight.airline} ${flight.flightNumber} - ${flight.price} ${flight.currency}`,
        timestamp: new Date(),
        type: 'system'
      };
      
      setComments(prev => [...prev, flightComment]);
    }
  };

  const handleAddCustomFlight = () => {
    console.log('Add custom flight');
    // Bu fonksiyon modal açabilir veya yeni sayfaya yönlendirebilir
  };

  const statusOptions = [
    { value: 'pending', label: 'Bekliyor' },
    { value: 'confirmed', label: 'Onaylandı' },
    { value: 'ticketed', label: 'Biletlendi' },
    { value: 'cancelled', label: 'İptal' },
    { value: 'rejected', label: 'Reddedildi' }
  ];

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-64"></div>
          <div className="h-64 bg-gray-200 rounded w-96"></div>
        </div>
      </div>
    );
  }

  if (!ticket) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Bilet bulunamadı</h2>
          <p className="text-gray-600">Aradığınız bilet mevcut değil.</p>
        </div>
      </div>
    );
  }

  return (
    <DetailLayout
      title={`Bilet Talebi #${ticket.id}`}
      subtitle={`${ticket.airline} ${ticket.flightNumber} - ${ticket.departureAirport} → ${ticket.arrivalAirport}`}
      backUrl="/tickets"
      recordId={ticket.id}
      recordType="ticket"
      comments={comments}
      onAddComment={handleAddComment}
      onStatusChange={handleStatusChange}
      statusOptions={statusOptions}
      currentStatus={ticket.status}
    >
      <div className="space-y-6">
        {/* Status Card */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Talep Durumu</h3>
            {renderStatusBadge(ticket.status, ticketStatusConfig)}
          </div>
          <div className="grid grid-cols-2 gap-4 text-sm">
            <div>
              <span className="text-gray-500">Talep Tarihi:</span>
              <p className="font-medium">{ticket.requestDate.toLocaleDateString('tr-TR')}</p>
            </div>
            <div>
              <span className="text-gray-500">Son Güncelleme:</span>
              <p className="font-medium">{ticket.updatedAt.toLocaleDateString('tr-TR')}</p>
            </div>
          </div>
        </div>

        {/* Customer Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <User className="h-5 w-5 mr-2" />
            Müşteri Bilgileri
          </h3>
          {renderCustomer(ticket.customer)}
          <div className="mt-4 grid grid-cols-2 gap-4 text-sm">
            <div>
              <span className="text-gray-500">E-posta:</span>
              <p className="font-medium">{ticket.customer.email}</p>
            </div>
            <div>
              <span className="text-gray-500">Telefon:</span>
              <p className="font-medium">{ticket.customer.phone}</p>
            </div>
          </div>
        </div>

        {/* Flight Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Plane className="h-5 w-5 mr-2" />
            Uçuş Bilgileri
          </h3>
          <div className="grid grid-cols-2 gap-6">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Kalkış</span>
              </div>
              <p className="text-lg font-semibold">{ticket.departureAirport}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Varış</span>
              </div>
              <p className="text-lg font-semibold">{ticket.arrivalAirport}</p>
            </div>
          </div>
          
          <div className="mt-6 grid grid-cols-3 gap-4">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Tarih</span>
              </div>
              <p className="font-medium">{ticket.departureDate.toLocaleDateString('tr-TR')}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Clock className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Saat</span>
              </div>
              <p className="font-medium">{ticket.departureTime}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Plane className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Uçuş</span>
              </div>
              <p className="font-medium">{ticket.airline} {ticket.flightNumber}</p>
            </div>
          </div>

          {ticket.pnr && (
            <div className="mt-4 p-3 bg-blue-50 border border-blue-200 rounded-lg">
              <div className="flex items-center space-x-2">
                <FileText className="h-4 w-4 text-blue-600" />
                <span className="text-sm font-medium text-blue-900">PNR:</span>
                <span className="font-mono text-blue-800">{ticket.pnr}</span>
              </div>
            </div>
          )}
        </div>

        {/* Passengers */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Yolcu Bilgileri</h3>
          <div className="space-y-4">
            {ticket.passengers.map((passenger, index) => (
              <div key={passenger.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                <div>
                  <p className="font-medium text-gray-900">
                    {passenger.firstName} {passenger.lastName}
                    {ticket.passengers.length > 1 && (
                      <span className="ml-2 px-2 py-1 text-xs bg-blue-100 text-blue-800 rounded">
                        Yolcu {index + 1}
                      </span>
                    )}
                  </p>
                  <p className="text-sm text-gray-500">{passenger.email}</p>
                </div>
                <div className="text-right">
                  <p className="text-sm text-gray-500">{passenger.phone}</p>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Price Info */}
        {ticket.ticketPrice && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
              <CreditCard className="h-5 w-5 mr-2" />
              Fiyat Bilgileri
            </h3>
            <div className="flex items-center justify-between">
              <span className="text-gray-600">Bilet Fiyatı:</span>
              <span className="text-2xl font-bold text-green-600">
                {ticket.ticketPrice.toLocaleString()} {ticket.currency}
              </span>
            </div>
          </div>
        )}

        {/* Agent Notes */}
        {ticket.agentNotes && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Agent Notları</h3>
            <p className="text-gray-700">{ticket.agentNotes}</p>
          </div>
        )}

        {/* Actions */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">İşlemler</h3>
          <div className="flex flex-wrap gap-3">
            <button className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
              <Edit className="h-4 w-4 mr-2" />
              Düzenle
            </button>
            
            {ticket.status === 'pending' && (
              <>
                <button 
                  onClick={() => handleStatusChange('confirmed')}
                  className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700"
                >
                  <CheckCircle className="h-4 w-4 mr-2" />
                  Onayla
                </button>
                <button 
                  onClick={() => handleStatusChange('rejected')}
                  className="flex items-center px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700"
                >
                  <XCircle className="h-4 w-4 mr-2" />
                  Reddet
                </button>
              </>
            )}
            
            {ticket.status === 'confirmed' && (
              <button 
                onClick={() => handleStatusChange('ticketed')}
                className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700"
              >
                <FileText className="h-4 w-4 mr-2" />
                Biletlendir
              </button>
            )}
            
            {ticket.pnr && (
              <button className="flex items-center px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700">
                <Download className="h-4 w-4 mr-2" />
                Bilet İndir
              </button>
            )}
          </div>
        </div>
      </div>
    </DetailLayout>
  );
}